/* fast_timer.h - run the 8253-style timer chip at a faster rate, so
 * we can schedule timer tasks with finer granularity.
 *	Mark Carson NIST/UMCP
 *	1/97
 */

/* Old latch value (for 100Hz clock) was 11932 = 2*2*19*157.  We use
 * even factors to avoid issues of rounding.  7600Hz gives us a clock
 * period of 131.6usec, more than fine enough granularity for our
 * purposes.
 *
 * On machines where the faster interrupts can't be maintained, use a smaller
 * factor.  Possibilities:
 *
 * Factor	Granularity (in usec)
 *     1	10000
 *     2	 5000
 *     4	 2500
 *    19	 2500
 *    38	  263.2
 *    76	  131.6
 *   157	   63.7
 *   314	   31.8
 *   628	   15.9
 *  2983	    3.4
 *
 * I have successfully run with clock rates up to 62800Hz (15.9usec) on
 * 200MHz machines.  Actually, I've run 298300Hz (3.4usec) clocks on
 * these machines, but feel that's a little excessive.
 */

#define FAST_FACTOR	76
#define FAST_HZ		FAST_FACTOR*HZ	/* number of fast ticks per second */

/* Convert between microseconds and "minijiffies" (fast timer ticks) */
#define HZ_TO_USEC	(1000000/HZ)

/* Careful to avoid integer overflow! */
extern inline int usec_to_minijiffy(int usec)
{
	return (usec/HZ_TO_USEC) * FAST_FACTOR +
		(((usec%HZ_TO_USEC)*FAST_FACTOR)+(HZ_TO_USEC/2))/HZ_TO_USEC;
}

extern inline int minijiffy_to_usec(int minijiffy)
{
	return (minijiffy/FAST_FACTOR)*HZ_TO_USEC +
		(((minijiffy%FAST_FACTOR)*HZ_TO_USEC)+(FAST_FACTOR/2))/FAST_FACTOR;
}

/* This is actually the same as the timer list structure, but I made a
 * separate one to help keep track of where I'm using it (with the hope
 * of moving everything to a loadable module at some point).
 */
struct fast_timer_list {
	struct fast_timer_list *next;
	struct fast_timer_list *prev;
	unsigned long expires;
	unsigned long data;
	void (*function)(struct fast_timer_list *);
};

extern void do_fast_timer(struct pt_regs *regs);
extern int add_fast_timer(struct fast_timer_list *timer);
extern int del_fast_timer(struct fast_timer_list *timer);

extern inline void init_fast_timer(struct fast_timer_list * timer)
{
	timer->next = NULL;
	timer->prev = NULL;
}
