/*
 *  linux/arch/i386/kernel/fast_timer.c
 *
 * Imitation being the sincerest form of flattery, this code is
 * patterned after (and reuses data structures from) the ordinary
 * Linux timer code.  The difference here is that times are kept
 * in mini-jiffies (currently 131.6 usec.)
 *
 *	Mark Carson, NIST
 *
 * $Header: /home/cvs/nistnet/patch/2.0.xx.fast.tar,v 1.2 2005/04/14 12:26:01 cvs Exp $
 */
#include <linux/errno.h>
#include <linux/sched.h>
#include <linux/kernel.h>
#include <linux/param.h>
#include <linux/string.h>
#include <linux/mm.h>
#include <linux/interrupt.h>

#include <asm/segment.h>
#include <asm/io.h>
#include <asm/irq.h>

#include <linux/mc146818rtc.h>
#include <linux/timex.h>
#include <linux/config.h>

#ifdef BREAK_DEBUG
#define BREAKPOINT() asm("   int $3");
#else
#define BREAKPOINT()
#endif

#ifdef CONFIG_FAST_TIMER
#include <linux/fast_timer.h>

#define MAX_TIME 2147483647	/* 2^31-1*/

/* This stuff was all local to sched.c, so I can't get it with a header
 * file.  Probably that's just as well, since it makes this code slightly
 * less vulnerable to breakage if I don't keep up with the frenetic pace
 * of Linux kernel development... - Mark
 */
#define TVN_BITS 6
#define TVR_BITS 8
#define TVN_SIZE (1 << TVN_BITS)
#define TVR_SIZE (1 << TVR_BITS)
#define TVN_MASK (TVN_SIZE - 1)
#define TVR_MASK (TVR_SIZE - 1)

struct fast_timer_vec {
        int index;
	struct fast_timer_list *vec[TVN_SIZE];
};

struct fast_timer_vec_root {
	int index;
	struct fast_timer_list *vec[TVR_SIZE];
};

static struct fast_timer_vec fast_tv5 = { 0 };
static struct fast_timer_vec fast_tv4 = { 0 };
static struct fast_timer_vec fast_tv3 = { 0 };
static struct fast_timer_vec fast_tv2 = { 0 };
static struct fast_timer_vec_root fast_tv1 = { 0 };

static struct fast_timer_vec * const fast_tvecs[] = {
        (struct fast_timer_vec *)&fast_tv1, &fast_tv2, &fast_tv3,
	&fast_tv4, &fast_tv5
};

#define NOOF_TVECS (sizeof(fast_tvecs) / sizeof(fast_tvecs[0]))

static unsigned long timer_minijiffies = 0, minijiffies = 0;

/*
 * In certain circumstances which I don't quite recall, I found it
 * was possible for an add operation to interrupt other list processing.
 * So, as a hack to get around this, I have a "hold queue" where adds
 * may be temporarily placed for later processing.  Actually, with
 * the vector used for holding timers in the new code, it's fairly
 * unlikely an add would tread on a list someone else is working on,
 * but better safe than sorry...
 */
static struct fast_timer_list *hold_queue;
#define HOLDING		2
#define EMPTYING	1	/* not really needed right now, but... */
#define EMPTY		0
static int hold_queue_state=EMPTY;

static inline void insert_fast_timer(struct fast_timer_list *timer,
                                struct fast_timer_list **vec, int idx)
{
	if ((timer->next = vec[idx]))
		vec[idx]->prev = timer;
	vec[idx] = timer;
	timer->prev = (struct fast_timer_list *)&vec[idx];
}

static inline void internal_add_fast_timer(struct fast_timer_list *timer)
{
	/*
	 * must be cli-ed when calling this
	 */
	unsigned long expires = timer->expires;
	unsigned long idx = expires - timer_minijiffies;

	if (idx < TVR_SIZE) {
		int i = expires & TVR_MASK;
		insert_fast_timer(timer, fast_tv1.vec, i);
	} else if (idx < 1 << (TVR_BITS + TVN_BITS)) {
		int i = (expires >> TVR_BITS) & TVN_MASK;
		insert_fast_timer(timer, fast_tv2.vec, i);
	} else if (idx < 1 << (TVR_BITS + 2 * TVN_BITS)) {
		int i = (expires >> (TVR_BITS + TVN_BITS)) & TVN_MASK;
		insert_fast_timer(timer, fast_tv3.vec, i);
	} else if (idx < 1 << (TVR_BITS + 3 * TVN_BITS)) {
		int i = (expires >> (TVR_BITS + 2 * TVN_BITS)) & TVN_MASK;
		insert_fast_timer(timer, fast_tv4.vec, i);
	} else if ((signed long) idx < 0) {
		/* can happen if you add a timer with expires == 0,
		 * or you set a timer to go off in the past
		 */
		insert_fast_timer(timer, fast_tv1.vec, fast_tv1.index);
	} else if (idx <= 0xffffffffUL) {
		int i = (expires >> (TVR_BITS + 3 * TVN_BITS)) & TVN_MASK;
		insert_fast_timer(timer, fast_tv5.vec, i);
	} else {
		/* Can only get here on architectures with 64-bit minijiffies */
		timer->next = timer->prev = timer;
	}
}

int add_fast_timer(struct fast_timer_list *timer)
{
	unsigned long flags;

	if (timer->next || timer->prev) {
		printk("add_fast_timer() called with non-zero list from %p\n",
		       __builtin_return_address(0));
		/* This doesn't seem worth crying over too much */
		/*return -EINVAL;*/
	}

	/* Check if too much time is requested */
	if (timer->expires > MAX_TIME) {
		return -EINVAL;
	}
	/* We're old-fashioned, and put in a relative, rather than
	 * absolute time.
	 */
	timer->expires += timer_minijiffies;

	save_flags(flags);
	cli();
	/* Check whether adds are on hold.  If so, add to hold queue
	 * rather than the real one.
	 */
	if (hold_queue_state == HOLDING) {
BREAKPOINT();	/* try to figure out how this can happen!!!!!!! */
		timer->next = hold_queue;
		hold_queue = timer;
	} else {
		internal_add_fast_timer(timer);
	}
	restore_flags(flags);
	return 0;
}

static inline int detach_fast_timer(struct fast_timer_list *timer)
{
	struct fast_timer_list *prev = timer->prev;
	if (prev) {
		struct fast_timer_list *next = timer->next;
		prev->next = next;
		if (next)
			next->prev = prev;
		return 1;
	}
	return 0;
}

/* I don't actually use this one... */
void mod_fast_timer(struct fast_timer_list *timer, unsigned long expires)
{
	unsigned long flags;

	timer->expires = expires;
	save_flags(flags);
	cli();
	detach_fast_timer(timer);
	internal_add_fast_timer(timer);
	restore_flags(flags);
}

int del_fast_timer(struct fast_timer_list * timer)
{
	int ret;
	unsigned long flags;

	save_flags(flags);
	cli();
	ret = detach_fast_timer(timer);
	timer->next = timer->prev = 0;
	restore_flags(flags);
	return ret;
}

static inline void cascade_fast_timers(struct fast_timer_vec *tv)
{
	/* cascade all the timers from tv up one level */
	struct fast_timer_list *timer;
	timer = tv->vec[tv->index];
	/*
	 * We are removing _all_ timers from the list, so we don't  have to
	 * detach them individually, just clear the list afterwards.
	 */
	while (timer) {
		struct fast_timer_list *tmp = timer;
		timer = timer->next;
		internal_add_fast_timer(tmp);
	}
	tv->vec[tv->index] = NULL;
	tv->index = (tv->index + 1) & TVN_MASK;
}

/* In case you're wondering, I commented out the sti/cli around the fn()
 * call because this can wreak havoc on the poor timer function which
 * gets interrupted before it can turn interrupts off...
 */
static inline void run_fast_timer_list(void)
{
	cli();
	/* Defer any adds while we process the list */
	hold_queue_state = HOLDING;
	while ((long)(minijiffies - timer_minijiffies) >= 0) {
		struct fast_timer_list *timer;
		if (!fast_tv1.index) {
			int n = 1;
			do {
				cascade_fast_timers(fast_tvecs[n]);
			} while (fast_tvecs[n]->index == 1 && ++n < NOOF_TVECS);
		}
		while ((timer = fast_tv1.vec[fast_tv1.index])) {
			void (*fn)(struct fast_timer_list *) = timer->function;
			detach_fast_timer(timer);
			timer->next = timer->prev = NULL;
			/*@@ sti(); */
			fn(timer);
			/*@@ cli(); */
		}
		++timer_minijiffies;
		fast_tv1.index = (fast_tv1.index + 1) & TVR_MASK;
	}
	/* Handle any deferred adds */
	hold_queue_state = EMPTYING;
	while (hold_queue) {
		struct fast_timer_list *timer;

BREAKPOINT();	/* try to figure out how this can happen!!!!!!! */

		timer = hold_queue;
		hold_queue = hold_queue->next;
		timer->prev = timer->next = NULL;
		(void) add_fast_timer(timer);
	}
	hold_queue_state = EMPTY;
	sti();
}

void do_fast_timer(struct pt_regs * regs)
{
        (*(unsigned long *)&minijiffies)++;
	/* Think about lost ticks, user mode... */
#ifdef notdef
        lost_miniticks++;
#endif
        run_fast_timer_list();
#ifdef notdef
        if (