#include <gtk/gtk.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <errno.h>
#include <malloc.h>
#include <memory.h>
#include "iet.h"
#include "classMainWindow.hpp"

//--------------------------------------------------------------------------------------------
// isNum();
//--------------------------------------------------------------------------------------------
int classMainWindow::isNum(const char *string) {
  const char *ptr=string;
  while(*ptr>='0' && *ptr<='9') ptr++;
  if(!*ptr)
    return 1;
  return 0;
}
//--------------------------------------------------------------------------------------------
// makePortNum(); -- return in NETWORK byte order THE Correct (tm) ermm.. that is little endian.
//--------------------------------------------------------------------------------------------
int classMainWindow::makePortNum(const char *string, const char* proto) {
  if(isNum(string))
    return(htons(atoi(string)));
  if(!strcmp(string,"any"))
    return 0;
  struct servent *service=getservbyname(string,proto);
  if (!service) {
    fprintf(stderr,"ERROR: No service for protocol \"%s\" by name:\"%s\"\n",proto,string);
    return 0;
  }
  return((service->s_port)); //From network byte order to host byte order
}
//--------------------------------------------------------------------------------------------
// makeProtocolNum();
//--------------------------------------------------------------------------------------------
int classMainWindow::makeProtocolNum(const char *string) {
  if(isNum(string))
    return(atoi(string));
  if(!strcmp(string,"any"))
    return 0;
  struct protoent *proto=getprotobyname(string);
  if (!proto) {
    fprintf(stderr,"ERROR: No protocol by name:\"%s\"\n",string);
    return 0;
  }
  return(proto->p_proto);
}
//--------------------------------------------------------------------------------------------
// makePortString();
//--------------------------------------------------------------------------------------------
void classMainWindow::makePortString(char *string, int size, int port,const char* proto) {
  if (!GTK_TOGGLE_BUTTON(gwStringsOnOff)->active) {
    snprintf(string,size,"%d",ntohs(port));
    return;
  }

  if (!port) {
    snprintf(string,size,"any");
    return;
  }

  struct servent *portent=NULL;
  portent=getservbyport(port,proto); // call it with network byte order
  if(!portent) {
    snprintf(string,size,"%d",port);
    return;
  }

  snprintf(string,size,"%s",portent->s_name);
  return;
}

//--------------------------------------------------------------------------------------------
// makeProtocolString();
//--------------------------------------------------------------------------------------------
void classMainWindow::makeProtocolString(char *string, int size, int protocol) {
  if (!GTK_TOGGLE_BUTTON(gwStringsOnOff)->active) {
    snprintf(string,size,"%d",protocol);
    return;
  }

  struct protoent *proto=NULL;
  proto=getprotobynumber(protocol);
  if(!proto) {
    snprintf(string,size,"%d",protocol);
    return;
  }

  snprintf(string,size,"%s",proto->p_name);
  return;
}
//--------------------------------------------------------------------------------------------
// updateRow();
//--------------------------------------------------------------------------------------------
int classMainWindow::updateRow() {
  NistnetTableEntry entry;
  struct in_addr inetAddx;
  int rc;
  memset(&entry,0x00,sizeof(entry)); // nastavimo vse na 0

  entry.lteProtocol=makeProtocolNum(gtk_entry_get_text(GTK_ENTRY(gwEntryProto)));
  rc=inet_aton(gtk_entry_get_text(GTK_ENTRY(gwEntrySrcIp)),&inetAddx);
  if(!rc) {
    printf("Warning, address not correctly formulated, aborting action! %d -1\n",inetAddx.s_addr);
    updateList();
    return 0;
  }
  entry.lteSource=inetAddx.s_addr;
  rc=inet_aton(gtk_entry_get_text(GTK_ENTRY(gwEntryDstIp)),&inetAddx);
  if(!rc) {
    printf("Warning, address not correctly formulated, aborting action! %d -2\n",inetAddx.s_addr);
    updateList();
    return 0;
  }
  entry.lteDest=inetAddx.s_addr;
  entry.lteSourceShortPort=makePortNum(gtk_entry_get_text(GTK_ENTRY(gwEntrySrcPort)),
                                       gtk_entry_get_text(GTK_ENTRY(gwEntryProto))
                                      );
  entry.lteDestShortPort=makePortNum(gtk_entry_get_text(GTK_ENTRY(gwEntryDstPort)),
                                     gtk_entry_get_text(GTK_ENTRY(gwEntryProto))
                                    );
  entry.lteBandwidth=atoi(gtk_entry_get_text(GTK_ENTRY(gwEntryBW)));

  double mu=atof(gtk_entry_get_text(GTK_ENTRY(gwEntryDelay)))*1000;
  double sigma=0;
  double rho=0;
  MakeDistributedStats(mu,sigma,rho,&entry.lteIDelay);

  entry.lteDrop=defFloatToInternal(atof(gtk_entry_get_text(GTK_ENTRY(gwEntryDrop))))/100;
  entry.lteDup=defFloatToInternal(atof(gtk_entry_get_text(GTK_ENTRY(gwEntryDup))))/100;
  entry.lteChangePacket=defFloatToInternal(atof(gtk_entry_get_text(GTK_ENTRY(gwEntryChangePacket))))/100;
  entry.lteChangeByte=defFloatToInternal(atof(gtk_entry_get_text(GTK_ENTRY(gwEntryChangeByte))))/100;
  if (GTK_TOGGLE_BUTTON(gwCheckProtocol)->active)
    entry.lteChangeFlags|=defCorruptProtoPayload;
  if (GTK_TOGGLE_BUTTON(gwCheckChecksum)->active)
    entry.lteChangeFlags|=defRecalculateCsum;

  if(nist->add(&entry)<0) {
    printf("Error, can't insert kernel entry. Are you sure you have enough permission for this operation?\n");
    gtk_exit(-1);
  }
  updateList();
  return 0;
}
//--------------------------------------------------------------------------------------------
// removeRow();
//--------------------------------------------------------------------------------------------
int classMainWindow::removeRow() {
  NistnetTableEntry entry;
  NistnetTableKey *key;
  if (selectedRow<0) // no selection
    return 0;
  key=(NistnetTableKey*)gtk_clist_get_row_data(GTK_CLIST(gwClist),selectedRow);
  if (!key) {
    printf("Error, no data associated with a row!\n");
    gtk_exit(1);
  }

  entry.lteKey=*key; // set the search key
  if (nist->statistics(&entry)) { // retreive data
    printf("Error, unable to retreive data!\n");
    gtk_exit(1);
  }

  if (nist->remove(&entry)) { // delete data
    printf("Error, unable to delete kernel entry!\n");
    gtk_exit(1);
  }
  updateList();
  return 0;
}
//--------------------------------------------------------------------------------------------
// updateForm();
//--------------------------------------------------------------------------------------------
// Read entries from
int classMainWindow::updateForm(int row) {
  print_debug("updateForm\n");
  struct in_addr temp;
  char proto[256];
  char buffer[256];
  NistnetTableEntry entry;
  NistnetTableKey *key;

  if(row==-1)
    return clearForm();

  key=(NistnetTableKey*)gtk_clist_get_row_data(GTK_CLIST(gwClist),row);

  if (!key) {
    printf("Error, no data associated with a row!\n");
    gtk_exit(1);
  }

  entry.lteKey=*key; // set the search key
  if (nist->statistics(&entry)) { // retreive data
    printf("Error, unable to retreive data!\n");
    gtk_exit(1);
  }

  makeProtocolString(proto,256,entry.lteProtocol);
  gtk_entry_set_text(GTK_ENTRY(gwEntryProto),proto);
  temp.s_addr=entry.lteKey.ltkSource;
  gtk_entry_set_text(GTK_ENTRY(gwEntrySrcIp),inet_ntoa(temp));
  temp.s_addr=entry.lteKey.ltkDest;
  gtk_entry_set_text(GTK_ENTRY(gwEntryDstIp),inet_ntoa(temp));

  makePortString(buffer,256,entry.lteSourceShortPort,proto);
  gtk_entry_set_text(GTK_ENTRY(gwEntrySrcPort),buffer);

  makePortString(buffer,256,entry.lteDestShortPort,proto);
  gtk_entry_set_text(GTK_ENTRY(gwEntryDstPort),buffer);

  snprintf(buffer,256,"%d",entry.lteBandwidth);
  gtk_entry_set_text(GTK_ENTRY(gwEntryBW),buffer);

  double mu=0;
  double sigma=0;
  double rho=0;
  UnmakeDistributedStats(&entry.lteIDelay,&mu,&sigma,&rho);
  mu/=1000;
  snprintf(buffer,256,"%.3f",mu);
  gtk_entry_set_text(GTK_ENTRY(gwEntryDelay),buffer);

  snprintf(buffer,256,"%.3f",defInternalToFloat(entry.lteDrop)*100);
  gtk_entry_set_text(GTK_ENTRY(gwEntryDrop),buffer);
  snprintf(buffer,256,"%.3f",defInternalToFloat(entry.lteDup)*100);
  gtk_entry_set_text(GTK_ENTRY(gwEntryDup),buffer);
  snprintf(buffer,256,"%.3f",defInternalToFloat(entry.lteChangePacket)*100);
  gtk_entry_set_text(GTK_ENTRY(gwEntryChangePacket),buffer);
  snprintf(buffer,256,"%.3f",defInternalToFloat(entry.lteChangeByte)*100);
  gtk_entry_set_text(GTK_ENTRY(gwEntryChangeByte),buffer);

  if (entry.lteChangeFlags & defCorruptProtoPayload)
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(gwCheckProtocol),TRUE);
  if (entry.lteChangeFlags & defRecalculateCsum)
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(gwCheckChecksum),TRUE);

  return 0;
}
//--------------------------------------------------------------------------------------------
// clearForm();
//--------------------------------------------------------------------------------------------
// Read entries from
int classMainWindow::clearForm() {
  print_debug("clearForm\n");
  gtk_entry_set_text(GTK_ENTRY(gwEntryProto),"");
  gtk_entry_set_text(GTK_ENTRY(gwEntrySrcIp),"");
  gtk_entry_set_text(GTK_ENTRY(gwEntrySrcPort),"");
  gtk_entry_set_text(GTK_ENTRY(gwEntryDstIp),"");
  gtk_entry_set_text(GTK_ENTRY(gwEntryDstPort),"");
  gtk_entry_set_text(GTK_ENTRY(gwEntryBW),"");
  gtk_entry_set_text(GTK_ENTRY(gwEntryDelay),"");
  gtk_entry_set_text(GTK_ENTRY(gwEntryDrop),"");
  gtk_entry_set_text(GTK_ENTRY(gwEntryDup),"");
  gtk_entry_set_text(GTK_ENTRY(gwEntryChangePacket),"");
  gtk_entry_set_text(GTK_ENTRY(gwEntryChangeByte),"");
  return 0;
}
//--------------------------------------------------------------------------------------------
// updateOnOff();
//--------------------------------------------------------------------------------------------
// Read entries from
int classMainWindow::updateOnOff() {
  print_debug("updateOnOff\n");
  struct nistnet_globalstats globs;
  nist->globalstat(&globs);
  if (globs.emulator_on) {
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(gwOnOff),TRUE);
    gtk_label_set_text(GTK_LABEL(GTK_BIN(gwOnOff)->child)," Nistnet is On ");
  } else {
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(gwOnOff),FALSE);
    gtk_label_set_text(GTK_LABEL(GTK_BIN(gwOnOff)->child)," Nistnet is Off ");
  }
  return 0;
}
//--------------------------------------------------------------------------------------------
// updateList();
//--------------------------------------------------------------------------------------------
// Read entries from
int classMainWindow::updateList() {
  print_debug("updateList\n");
  // fetch data
  int count,dataCount,row;
  NistnetTableKey *data;
  NistnetTableKey *key;

  for(count=defNistnetDataPrefetch;;count+=defNistnetDataPrefetchIncrement) {
    data=new NistnetTableKey[count];
    if (!data) {
      printf("Error, out of memory!\n");
      gtk_exit(1);
    }
    dataCount=nist->read(data,count);
    printf("Read %d\n",dataCount);
    if (count>dataCount) // we've read less data than there is space => we have all the data
      break;
    delete data;
  }

  // insert data into CList
  char buffer[256];
  char *strings[3]; //3x strings of 256bytes each
  strings[0]=new char[256];
  strings[1]=new char[256];
  strings[2]=new char[256];
  if(!strings[0] |
     !strings[1] |
     !strings[2] ) {
    printf("Error, out of memory!\n");
    gtk_exit(1);
  }

  gtk_clist_freeze(GTK_CLIST(gwClist));
  gtk_clist_clear(GTK_CLIST(gwClist));

  for(int i=0;i<dataCount;i++) {
    struct in_addr temp;
    makeProtocolString(strings[0],256,data[i].ltkProtocol);
    makePortString(buffer,256,data[i].ltkSourcePort.ltp.ltpPort,strings[0]);
    temp.s_addr=data[i].ltkSource;
    snprintf(strings[1],256,"%s:%s",
             temp.s_addr ? inet_ntoa(temp) : "inaddr_any",
             buffer
            );
    makePortString(buffer,256,data[i].ltkDestPort.ltp.ltpPort,strings[0]);
    temp.s_addr=data[i].ltkDest;
    snprintf(strings[2],256,"%s:%s",
             temp.s_addr ? inet_ntoa(temp) : "inaddr_any",
             buffer
            );
    row=gtk_clist_append(GTK_CLIST(gwClist),strings);
    key=(NistnetTableKey*)malloc(sizeof(*key));
    if (!key) {
      printf("Error, out of memory!\n");
      gtk_exit(1);
    }
    *key=data[i];
    // associate row with a pointer and a free(). Thanx god for "C". :-)
    gtk_clist_set_row_data_full(GTK_CLIST(gwClist),row,key,free);
  }/*for*/
  gtk_clist_thaw(GTK_CLIST(gwClist));

  delete strings[0];
  delete strings[1];
  delete strings[2];
  delete data;
  selectedRow=-1;
  return 0;
}
//--------------------------------------------------------------------------------------------
// gsignalUpdate(...);
//--------------------------------------------------------------------------------------------
void classMainWindow::gsignalUpdate(GtkWidget *widget, classMainWindow *data) {
  print_debug("->gsignalAdd!\n");
  data->updateRow();
}
//--------------------------------------------------------------------------------------------
// gsignalRemove(...);
//--------------------------------------------------------------------------------------------
void classMainWindow::gsignalRemove(GtkWidget *widget, classMainWindow *data) {
  print_debug("->gsignalRemove!\n");
  data->removeRow();
}
//--------------------------------------------------------------------------------------------
// gsignalSelected(...);
//--------------------------------------------------------------------------------------------
void classMainWindow::gsignalSelected(GtkWidget *widget,
                                      gint row, gint col,
                                      GdkEventButton *event,
                                      classMainWindow *data
                                     )
{
  print_debug("->gsignalSelected!\n");
  data->selectedRow=row;
  data->updateForm(row);
}
//--------------------------------------------------------------------------------------------
// gsignalUnSelected(...);
//--------------------------------------------------------------------------------------------
void classMainWindow::gsignalUnSelected(GtkWidget *widget,
                                        gint row, gint col,
                                        GdkEventButton *event,
                                        classMainWindow *data
                                       )
{
  print_debug("->gsignalUnSelected!\n");
  data->clearForm();
  data->selectedRow=-1;
}
//--------------------------------------------------------------------------------------------
// gsignalDestroy(...);
//--------------------------------------------------------------------------------------------
void classMainWindow::gsignalDestroy(GtkWidget *widget, classMainWindow *data) {
  print_debug("->gsignalDestroy!\n");
  gtk_main_quit();
}
//--------------------------------------------------------------------------------------------
// gsignalReread(...);
//--------------------------------------------------------------------------------------------
void classMainWindow::gsignalReread(GtkWidget *widget, classMainWindow *data) {
  print_debug("->gsignalReread!\n");
  data->updateList();
  data->updateOnOff();
}
//--------------------------------------------------------------------------------------------
// gsignalStringsOnOff(...);
//--------------------------------------------------------------------------------------------
void classMainWindow::gsignalStringsOnOff(GtkWidget *widget, classMainWindow *data) {
  print_debug("->gsignalStringsOnOff!\n");
  int row=data->selectedRow;
  data->updateList();
  gtk_clist_select_row(GTK_CLIST(data->gwClist),row,0);
}
//--------------------------------------------------------------------------------------------
// gsignalOnOff(...);
//--------------------------------------------------------------------------------------------
void classMainWindow::gsignalOnOff(GtkWidget *widget, classMainWindow *data) {
  print_debug("->gsignalOnOff!\n");
  if (GTK_TOGGLE_BUTTON(data->gwOnOff)->active) {
    data->nist->on();
    gtk_label_set_text(GTK_LABEL(GTK_BIN(data->gwOnOff)->child)," Nistnet is On ");
  } else {
    data->nist->off();
    gtk_label_set_text(GTK_LABEL(GTK_BIN(data->gwOnOff)->child)," Nistnet is Off ");
  }
}
//--------------------------------------------------------------------------------------------
// disable();
//--------------------------------------------------------------------------------------------
void classMainWindow::disable() {
  gtk_widget_set_sensitive(gwMainWindow,0);
}
//--------------------------------------------------------------------------------------------
// enable();
//--------------------------------------------------------------------------------------------
void classMainWindow::enable() {
  gtk_widget_set_sensitive(gwMainWindow,1);
}
//--------------------------------------------------------------------------------------------
// hide();
//--------------------------------------------------------------------------------------------
void classMainWindow::hide()
{
  gtk_widget_hide(gwMainWindow);
}
//--------------------------------------------------------------------------------------------
// show();
//--------------------------------------------------------------------------------------------
void classMainWindow::show()
{
  gtk_widget_show(gwMainWindow);
}
//--------------------------------------------------------------------------------------------
// constructor
//--------------------------------------------------------------------------------------------
classMainWindow::classMainWindow() {

  selectedRow=-1;

  nist=new classNistnetEngine;
  if (!nist) {
    fprintf(stderr,"Unable to create nistnet device object!\n");
    gtk_exit(1);
  }
  nist->open();

  // Create the main window
  gwMainWindow = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title(GTK_WINDOW(gwMainWindow),defAppName);
  gtk_container_set_border_width(GTK_CONTAINER(gwMainWindow),2);
  // gtk_widget_set_usize(GTK_WIDGET(gwMainWindow), 600, 400);
  gtk_window_position(GTK_WINDOW(gwMainWindow),GTK_WIN_POS_CENTER);
  gtk_signal_connect(GTK_OBJECT(gwMainWindow),
                     "destroy",
                     GTK_SIGNAL_FUNC(gsignalDestroy),
                     this
                    );
  gwHbox = gtk_hbox_new(FALSE,5);
  gtk_container_add (GTK_CONTAINER (gwMainWindow), gwHbox);
  gtk_widget_show(gwHbox);

  /* 1st VBOX */
  gwVbox1 = gtk_vbox_new(FALSE,5);
  gtk_box_pack_start (GTK_BOX (gwHbox), gwVbox1, TRUE, TRUE, 0);
  gtk_widget_show(gwVbox1);

  /* Scroll box for CList */
  gwClistScrWindow = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (gwClistScrWindow),
                                  GTK_POLICY_AUTOMATIC,
                                  GTK_POLICY_AUTOMATIC
                                 );
  gtk_box_pack_start (GTK_BOX (gwVbox1), gwClistScrWindow, TRUE, TRUE, 0);
  gtk_widget_show(gwClistScrWindow);

  char *strings[]={" Protocol ","  Source IP : Port  ","  Destination IP : Port  "};
  gwClist = gtk_clist_new_with_titles(3,strings);
  gtk_clist_set_selection_mode(GTK_CLIST(gwClist),GTK_SELECTION_SINGLE);
  gtk_clist_column_titles_passive(GTK_CLIST(gwClist));
  gtk_container_add(GTK_CONTAINER(gwClistScrWindow), gwClist);
  gtk_signal_connect(GTK_OBJECT(gwClist),
                     "select_row",
                     GTK_SIGNAL_FUNC(gsignalSelected),
                     this
                    );
  gtk_signal_connect(GTK_OBJECT(gwClist),
                     "unselect_row",
                     GTK_SIGNAL_FUNC(gsignalUnSelected),
                     this
                    );
  gtk_widget_show(gwClist);

    /* 1.1 HBOX */
    gwHbox11 = gtk_hbox_new(TRUE,5);
    gtk_box_pack_start (GTK_BOX (gwVbox1), gwHbox11, FALSE, FALSE, 0);
    gtk_widget_show(gwHbox11);

    gwOnOff = gtk_toggle_button_new_with_label(" Nistnet is Off ");
    gtk_box_pack_start (GTK_BOX (gwHbox11), gwOnOff, TRUE, TRUE, 0);
    gtk_signal_connect(GTK_OBJECT(gwOnOff),
                       "clicked",
                       GTK_SIGNAL_FUNC(gsignalOnOff),
                       this);
    gtk_widget_show(gwOnOff);

    gwStringsOnOff = gtk_toggle_button_new_with_label(" Strings On/Off ");
    gtk_box_pack_start (GTK_BOX (gwHbox11), gwStringsOnOff, TRUE, TRUE, 0);
    gtk_signal_connect(GTK_OBJECT(gwStringsOnOff),
                       "clicked",
                       GTK_SIGNAL_FUNC(gsignalStringsOnOff),
                       this);
    gtk_widget_show(gwStringsOnOff);

    gwReread = gtk_button_new_with_label(" Reread ");
    gtk_box_pack_start (GTK_BOX (gwHbox11), gwReread, TRUE, TRUE, 0);
    gtk_signal_connect(GTK_OBJECT(gwReread),
                       "clicked",
                       GTK_SIGNAL_FUNC(gsignalReread),
                       this);
    gtk_widget_show(gwReread);


  /* 2nd VBOX */
  gwVbox2 = gtk_vbox_new(FALSE,0);
  gtk_box_pack_start (GTK_BOX (gwHbox), gwVbox2, FALSE, FALSE, 0);
  gtk_widget_show(gwVbox2);

  gwLabelSrc=gtk_label_new("Source IP [xxx.xxx.xxx.xxx]");
  gtk_box_pack_start (GTK_BOX (gwVbox2), gwLabelSrc, FALSE, FALSE, 0);
  gtk_widget_show(gwLabelSrc);

    /* 2.1 HBOX */
    gwHbox21 = gtk_hbox_new(FALSE,0);
    gtk_box_pack_start (GTK_BOX (gwVbox2), gwHbox21, FALSE, FALSE, 0);
    gtk_widget_show(gwHbox21);

    gwEntrySrcIp=gtk_entry_new();
    gtk_box_pack_start (GTK_BOX (gwHbox21), gwEntrySrcIp, FALSE, FALSE, 0);
    gtk_widget_show(gwEntrySrcIp);

    gwLabelSrcPort=gtk_label_new(" Port ");
    gtk_box_pack_start (GTK_BOX (gwHbox21), gwLabelSrcPort, FALSE, FALSE, 0);
    gtk_widget_show(gwLabelSrcPort);

    gwEntrySrcPort=gtk_entry_new();
    gtk_box_pack_start (GTK_BOX (gwHbox21), gwEntrySrcPort, FALSE, FALSE, 0);
    gtk_widget_show(gwEntrySrcPort);

  gwLabelDst=gtk_label_new("Destination IP [xxx.xxx.xxx.xxx]");
  gtk_box_pack_start (GTK_BOX (gwVbox2), gwLabelDst, FALSE, FALSE, 0);
  gtk_widget_show(gwLabelDst);

    /* 2.2 HBOX */
    gwHbox22 = gtk_hbox_new(FALSE,0);
    gtk_box_pack_start (GTK_BOX (gwVbox2), gwHbox22, FALSE, FALSE, 0);
    gtk_widget_show(gwHbox22);

    gwEntryDstIp=gtk_entry_new();
    gtk_box_pack_start (GTK_BOX (gwHbox22), gwEntryDstIp, FALSE, FALSE, 0);
    gtk_widget_show(gwEntryDstIp);

    gwLabelDstPort=gtk_label_new(" Port ");
    gtk_box_pack_start (GTK_BOX (gwHbox22), gwLabelDstPort, FALSE, FALSE, 0);
    gtk_widget_show(gwLabelDstPort);

    gwEntryDstPort=gtk_entry_new();
    gtk_box_pack_start (GTK_BOX (gwHbox22), gwEntryDstPort, FALSE, FALSE, 0);
    gtk_widget_show(gwEntryDstPort);

  gwLabelProto=gtk_label_new("Protocol");
  gtk_box_pack_start (GTK_BOX (gwVbox2), gwLabelProto, FALSE, FALSE, 0);
  gtk_widget_show(gwLabelProto);

  gwEntryProto=gtk_entry_new();
  gtk_box_pack_start (GTK_BOX (gwVbox2), gwEntryProto, FALSE, FALSE, 0);
  gtk_widget_show(gwEntryProto);

  gwCheckProtocol=gtk_check_button_new_with_label("Corruption on protocol layer not on IP");
  gtk_box_pack_start (GTK_BOX (gwVbox2), gwCheckProtocol, FALSE, FALSE, 0);
  gtk_widget_show(gwCheckProtocol);

  gwCheckChecksum=gtk_check_button_new_with_label("Recalculate CRC of corrupted packets");
  gtk_box_pack_start (GTK_BOX (gwVbox2), gwCheckChecksum, FALSE, FALSE, 0);
  gtk_widget_show(gwCheckChecksum);

  gwLabelChangePacket=gtk_label_new("Packet corruption [%]");
  gtk_box_pack_start (GTK_BOX (gwVbox2), gwLabelChangePacket, FALSE, FALSE, 0);
  gtk_widget_show(gwLabelChangePacket);

  gwEntryChangePacket=gtk_entry_new();
  gtk_box_pack_start (GTK_BOX (gwVbox2), gwEntryChangePacket, FALSE, FALSE, 0);
  gtk_widget_show(gwEntryChangePacket);

  gwLabelChangeByte=gtk_label_new("Corruption of every single byte in a packet [%]");
  gtk_box_pack_start (GTK_BOX (gwVbox2), gwLabelChangeByte, FALSE, FALSE, 0);
  gtk_widget_show(gwLabelChangeByte);

  gwEntryChangeByte=gtk_entry_new();
  gtk_box_pack_start (GTK_BOX (gwVbox2), gwEntryChangeByte, FALSE, FALSE, 0);
  gtk_widget_show(gwEntryChangeByte);

  gwLabelBW=gtk_label_new("Bandwidth [Bps]");
  gtk_box_pack_start (GTK_BOX (gwVbox2), gwLabelBW, FALSE, FALSE, 0);
  gtk_widget_show(gwLabelBW);

  gwEntryBW=gtk_entry_new();
  gtk_box_pack_start (GTK_BOX (gwVbox2), gwEntryBW, FALSE, FALSE, 0);
  gtk_widget_show(gwEntryBW);

  gwLabelDelay=gtk_label_new("Delay [ms]");
  gtk_box_pack_start (GTK_BOX (gwVbox2), gwLabelDelay, FALSE, FALSE, 0);
  gtk_widget_show(gwLabelDelay);

  gwEntryDelay=gtk_entry_new();
  gtk_box_pack_start (GTK_BOX (gwVbox2), gwEntryDelay, FALSE, FALSE, 0);
  gtk_widget_show(gwEntryDelay);

  gwLabelDrop=gtk_label_new("Drop [%]");
  gtk_box_pack_start (GTK_BOX (gwVbox2), gwLabelDrop, FALSE, FALSE, 0);
  gtk_widget_show(gwLabelDrop);

  gwEntryDrop=gtk_entry_new();
  gtk_box_pack_start (GTK_BOX (gwVbox2), gwEntryDrop, FALSE, FALSE, 0);
  gtk_widget_show(gwEntryDrop);

  gwLabelDup=gtk_label_new("Dup [%]");
  gtk_box_pack_start (GTK_BOX (gwVbox2), gwLabelDup, FALSE, FALSE, 0);
  gtk_widget_show(gwLabelDup);

  gwEntryDup=gtk_entry_new();
  gtk_box_pack_start (GTK_BOX (gwVbox2), gwEntryDup, FALSE, FALSE, 0);
  gtk_widget_show(gwEntryDup);

    /* 2.4 HBOX */
    gwHbox24 = gtk_hbox_new(TRUE,5);
    gtk_box_pack_start (GTK_BOX (gwVbox2), gwHbox24, FALSE, FALSE, 0);
    gtk_widget_show(gwHbox24);

    gwButtonUpdate=gtk_button_new_with_label("Update");
    gtk_box_pack_start (GTK_BOX (gwHbox24), gwButtonUpdate, TRUE, TRUE, 0);
    gtk_signal_connect(GTK_OBJECT(gwButtonUpdate),
                       "clicked",
                       GTK_SIGNAL_FUNC(gsignalUpdate),
                       this);
    gtk_widget_show(gwButtonUpdate);

    gwButtonRemove=gtk_button_new_with_label("Remove");
    gtk_box_pack_start (GTK_BOX (gwHbox24), gwButtonRemove, TRUE, TRUE, 0);
    gtk_signal_connect(GTK_OBJECT(gwButtonRemove),
                       "clicked",
                       GTK_SIGNAL_FUNC(gsignalRemove),
                       this);
    gtk_widget_show(gwButtonRemove);

  updateList(); // refresh the list and check button.
  updateOnOff();
}
//--------------------------------------------------------------------------------------------
// destructor
//--------------------------------------------------------------------------------------------
classMainWindow::~classMainWindow() {
  nist->close();
  delete nist;
}
