/*
 *  linux/arch/i386/kernel/fast_timer.c
 *
 * Imitation being the sincerest form of flattery, this code is
 * patterned after (and reuses data structures from) the ordinary
 * Linux timer code.  The difference here is that times are kept
 * in mini-jiffies (currently 131.6 usec.)
 *
 *	Mark Carson, NIST
 *
 * $Header: /home/cvs/nistnet/patch/2.2.xx.fast.tar,v 1.2 2005/04/14 12:26:01 cvs Exp $
 */
#include <linux/errno.h>
#include <linux/sched.h>
#include <linux/kernel.h>
#include <linux/param.h>
#include <linux/string.h>
#include <linux/mm.h>
#include <linux/interrupt.h>

#include <asm/segment.h>
#include <asm/io.h>
#include <asm/irq.h>

#include <linux/mc146818rtc.h>
#include <linux/timex.h>
#include <linux/config.h>

#ifdef BREAK_DEBUG
#define BREAKPOINT() asm("   int $3");
#else
#define BREAKPOINT()
#endif

#ifdef CONFIG_FAST_TIMER
#include <linux/fast_timer.h>

#define MAX_TIME 2147483647	/* 2^31-1*/

/* This stuff was all local to sched.c, so I can't get it with a header
 * file.  Probably that's just as well, since it makes this code slightly
 * less vulnerable to breakage if I don't keep up with the frenetic pace
 * of Linux kernel development... - Mark
 */
#define TVN_BITS 6
#define TVR_BITS 8
#define TVN_SIZE (1 << TVN_BITS)
#define TVR_SIZE (1 << TVR_BITS)
#define TVN_MASK (TVN_SIZE - 1)
#define TVR_MASK (TVR_SIZE - 1)

struct fast_timer_vec {
        int index;
	struct fast_timer_list *vec[TVN_SIZE];
};

struct fast_timer_vec_root {
	int index;
	struct fast_timer_list *vec[TVR_SIZE];
};

static struct fast_timer_vec fast_tv5 = { 0 };
static struct fast_timer_vec fast_tv4 = { 0 };
static struct fast_timer_vec fast_tv3 = { 0 };
static struct fast_timer_vec fast_tv2 = { 0 };
static struct fast_timer_vec_root fast_tv1 = { 0 };

static struct fast_timer_vec * const fast_tvecs[] = {
        (struct fast_timer_vec *)&fast_tv1, &fast_tv2, &fast_tv3,
	&fast_tv4, &fast_tv5
};

#define NOOF_TVECS (sizeof(fast_tvecs) / sizeof(fast_tvecs[0]))

static unsigned long timer_minijiffies = 0, minijiffies = 0;

static inline void insert_fast_timer(struct fast_timer_list *timer,
                                struct fast_timer_list **vec, int idx)
{
	if ((timer->next = vec[idx]))
		vec[idx]->prev = timer;
	vec[idx] = timer;
	timer->prev = (struct fast_timer_list *)&vec[idx];
}

static inline void internal_add_fast_timer(struct fast_timer_list *timer)
{
	/*
	 * must be cli-ed when calling this
	 */
	unsigned long expires = timer->expires;
	unsigned long idx = expires - timer_minijiffies;

	if (idx < TVR_SIZE) {
		int i = expires & TVR_MASK;
		insert_fast_timer(timer, fast_tv1.vec, i);
	} else if (idx < 1 << (TVR_BITS + TVN_BITS)) {
		int i = (expires >> TVR_BITS) & TVN_MASK;
		insert_fast_timer(timer, fast_tv2.vec, i);
	} else if (idx < 1 << (TVR_BITS + 2 * TVN_BITS)) {
		int i = (expires >> (TVR_BITS + TVN_BITS)) & TVN_MASK;
		insert_fast_timer(timer, fast_tv3.vec, i);
	} else if (idx < 1 << (TVR_BITS + 3 * TVN_BITS)) {
		int i = (expires >> (TVR_BITS + 2 * TVN_BITS)) & TVN_MASK;
		insert_fast_timer(timer, fast_tv4.vec, i);
	} else if ((signed long) idx < 0) {
		/* can happen if you add a timer with expires == 0,
		 * or you set a timer to go off in the past
		 */
		insert_fast_timer(timer, fast_tv1.vec, fast_tv1.index);
	} else if (idx <= 0xffffffffUL) {
		int i = (expires >> (TVR_BITS + 3 * TVN_BITS)) & TVN_MASK;
		insert_fast_timer(timer, fast_tv5.vec, i);
	} else {
		/* Can only get here on architectures with 64-bit minijiffies */
		timer->next = timer->prev = timer;
	}
}

/*
 * OK, I guess I sort of predicted it, but that still doesn't mean I
 * entirely understand it.  However, it seems it's still possible,
 * despite what the locking is trying to do, for an add operation to
 * interrupt other list processing.  So to get around this, we implement
 * a "hold queue" for adds to be placed temporarily for later
 * processing.
 */

static struct fast_timer_list *hold_queue;
#define HOLDING		2
#define EMPTYING	1
#define EMPTY		0
static int hold_queue_state=EMPTY;

spinlock_t fast_timerlist_lock = SPIN_LOCK_UNLOCKED;

int add_fast_timer(struct fast_timer_list *timer)
{
	unsigned long flags;

	/* Check if too much time is requested */
	if (timer->expires > MAX_TIME)
		return -EINVAL;
	/* We're old-fashioned, and put in a relative, rather than
	 * absolute time.
	 */
	timer->expires += timer_minijiffies;

	spin_lock_irqsave(&fast_timerlist_lock, flags);

	if (timer->prev)
		goto bug;

	/* Check whether adds are on hold.  If so, add to the hold queue
	 * rather than the real one.
	 */
	switch (hold_queue_state) {
	case HOLDING:
		BREAKPOINT();	/* try to figure out how this happens!! */
		printk("bug: add_fast_timer added during hold at %p.\n",
				__builtin_return_address(0));
		timer->next = hold_queue;
		hold_queue = timer;
		break;
	case EMPTYING:
		/* This could be very serious, since we're clearing out
		 * the queue now.  Try adding to the end */
		{
		struct fast_timer_list *runner;

		BREAKPOINT();	/* try to figure out how this happens!! */
		printk("bug: add_fast_timer added during emptying at %p.\n",
				__builtin_return_address(0));
		runner = hold_queue;
		while (runner->next) {
			runner = runner->next;
		}
		timer->next = NULL;
		runner->next = timer;
		break;
		}
	default:
		internal_add_fast_timer(timer);
		break;
	}
out:
	spin_unlock_irqrestore(&fast_timerlist_lock, flags);
	return 0;

bug:
	printk("bug: kernel fast timer added twice at %p.\n",
			__builtin_return_address(0));
	goto out;
}

static inline int detach_fast_timer(struct fast_timer_list *timer)
{
	struct fast_timer_list *prev = timer->prev;
	if (prev) {
		struct fast_timer_list *next = timer->next;
		prev->next = next;
		if (next)
			next->prev = prev;
		return 1;
	}
	return 0;
}

/* I don't actually use this one... */
void mod_fast_timer(struct fast_timer_list *timer, unsigned long expires)
{
	unsigned long flags;

	spin_lock_irqsave(&fast_timerlist_lock, flags);
	timer->expires = expires;
	detach_fast_timer(timer);
	internal_add_fast_timer(timer);
	spin_unlock_irqrestore(&fast_timerlist_lock, flags);
}

int del_fast_timer(struct fast_timer_list * timer)
{
	int ret;
	unsigned long flags;

	spin_lock_irqsave(&fast_timerlist_lock, flags);
	ret = detach_fast_timer(timer);
	timer->next = timer->prev = 0;
	spin_unlock_irqrestore(&fast_timerlist_lock, flags);
	return ret;
}

static inline void cascade_fast_timers(struct fast_timer_vec *tv)
{
	/* cascade all the timers from tv up one level */
	struct fast_timer_list *timer;
	timer = tv->vec[tv->index];
	/*
	 * We are removing _all_ timers from the list, so we don't  have to
	 * detach them individually, just clear the list afterwards.
	 */
	while (timer) {
		struct fast_timer_list *tmp = timer;
		timer = timer->next;
		internal_add_fast_timer(tmp);
	}
	tv->vec[tv->index] = NULL;
	tv->index = (tv->index + 1) & TVN_MASK;
}

static inline void run_fast_timer_list(void)
{
	unsigned long flags;

	spin_lock_irq(&fast_timerlist_lock);

	/* Defer any adds while we process the list */
	hold_queue_state = HOLDING;

	while ((long)(minijiffies - timer_minijiffies) >= 0) {
		struct fast_timer_list *timer;
		if (!fast_tv1.index) {
			int n = 1;
			do {
				cascade_fast_timers(fast_tvecs[n]);
			} while (fast_tvecs[n]->index == 1 && ++n < NOOF_TVECS);
		}
		while ((timer = fast_tv1.vec[fast_tv1.index])) {
			void (*fn)(struct fast_timer_list *) = timer->function;
			detach_fast_timer(timer);
			timer->next = timer->prev = NULL;
			/*@@save_flags(flags);*/
			spin_unlock_irq(&fast_timerlist_lock);/*@@*/
			/*@@restore_flags(timer->flags);*/
			fn(timer);
			/*@@restore_flags(flags);*/
			spin_lock_irq(&fast_timerlist_lock);/*@@*/
		}
		++timer_minijiffies;
		fast_tv1.index = (fast_tv1.index + 1) & TVR_MASK;
	}

	/* Handle any deferred adds */
	hold_queue_state = EMPTYING;
	while (hold_queue) {
		struct fast_timer_list *timer;

		BREAKPOINT();	/* try to figure out how this can happen!! */
		printk("bug: run_fast_timer found adds during hold at %p.\n",
				__builtin_return_address(0));
		timer = hold_queue;
		hold_queue = hold_queue->next;
		timer->prev = timer->next = NULL;
		(void) internal_add_fast_timer(timer);
	}
	hold_queue_state = EMPTY;

	spin_unlock_irq(&fast_timerlist_lock);
}

void do_fast_timer(struct pt_regs * regs)
{
        (*(unsigned long *)&minijiffies)++;
	/* Think about lost ticks, user mode... */
#ifdef notdef
        lost_miniticks++;
#endif
        run_fast_timer_list();
#ifdef notdef
        if (